<?php

// This is the main LeadService used for working with forms & leads.
// It is designed to be used with manual requires and global classes,
// consistent with your current bootstrap/app.php setup.

use App\Models\Form;
use App\Models\Lead;
use PDO;
use PDOException;

class LeadService
{
    /** @var PDO */
    private $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Create a new form
     */
    public function createForm(string $name, array $fields): int
    {
        $sql  = "INSERT INTO forms (name, fields, created_at) VALUES (?, ?, NOW())";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$name, json_encode($fields)]);

        return (int) $this->pdo->lastInsertId();
    }

    /**
     * Get all forms
     */
    public function getForms(): array
    {
        $sql   = "SELECT * FROM forms ORDER BY created_at DESC";
        $stmt  = $this->pdo->query($sql);
        $forms = [];

        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Assuming you have App\Models\Form model class
            $forms[] = new Form($row);
        }

        return $forms;
    }

    /**
     * Get form by ID
     */
    public function getForm(int $id): ?Form
    {
        $sql  = "SELECT * FROM forms WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);

        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        return $data ? new Form($data) : null;
    }

    /**
     * Capture a new lead from form submission
     * Supports file uploads via $_FILES
     */
    public function captureLead(int $formId, array $data, array $files = []): int
    {
        // Handle file uploads if any
        $uploadedFiles = $this->handleFileUploads($files);
        $leadData      = array_merge($data, $uploadedFiles);

        $sql  = "INSERT INTO leads (form_id, data, created_at) VALUES (?, ?, NOW())";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$formId, json_encode($leadData)]);

        return (int) $this->pdo->lastInsertId();
    }

    /**
     * Handle file uploads from form submissions
     */
    private function handleFileUploads(array $files): array
    {
        $uploadedFiles = [];

        foreach ($files as $fieldName => $file) {
            if (
                isset($file['error'], $file['size']) &&
                $file['error'] === UPLOAD_ERR_OK &&
                $file['size'] > 0
            ) {
                $fileName = $this->generateSafeFileName($file['name']);
                $filePath = UPLOAD_PATH . '/' . $fileName;

                if (move_uploaded_file($file['tmp_name'], $filePath)) {
                    $uploadedFiles[$fieldName] = $fileName;
                }
            }
        }

        return $uploadedFiles;
    }

    /**
     * Generate safe filename for uploads
     */
    private function generateSafeFileName(string $originalName): string
    {
        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        $baseName  = pathinfo($originalName, PATHINFO_FILENAME);

        // Sanitize filename
        $safeName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $baseName);
        $safeName = substr($safeName, 0, 100); // Limit length

        return uniqid('', true) . '_' . $safeName . ($extension ? '.' . $extension : '');
    }

    /**
     * Get all leads with optional filtering
     * Filters: form_id, date_from (Y-m-d), date_to (Y-m-d)
     */
    public function getLeads(array $filters = []): array
    {
        $sql = "
            SELECT l.*, f.name AS form_name 
            FROM leads l 
            LEFT JOIN forms f ON l.form_id = f.id 
            WHERE 1=1
        ";
        $params = [];

        if (!empty($filters['form_id'])) {
            $sql     .= " AND l.form_id = ?";
            $params[] = $filters['form_id'];
        }

        if (!empty($filters['date_from'])) {
            $sql     .= " AND DATE(l.created_at) >= ?";
            $params[] = $filters['date_from'];
        }

        if (!empty($filters['date_to'])) {
            $sql     .= " AND DATE(l.created_at) <= ?";
            $params[] = $filters['date_to'];
        }

        $sql .= " ORDER BY l.created_at DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        $leads = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $leads[] = new Lead($row);
        }

        return $leads;
    }

    /**
     * Get single lead by ID
     */
    public function getLead(int $id): ?Lead
    {
        $sql = "
            SELECT l.*, f.name AS form_name 
            FROM leads l 
            LEFT JOIN forms f ON l.form_id = f.id 
            WHERE l.id = ?
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);

        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        return $data ? new Lead($data) : null;
    }

    /**
     * Update lead score and insights (AI analysis)
     */
    public function updateLeadAnalysis(int $leadId, int $score, string $insights): bool
    {
        $sql  = "UPDATE leads SET score = ?, insights = ? WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);

        return $stmt->execute([$score, $insights, $leadId]);
    }

    /**
     * Get dashboard statistics (simple global stats)
     */
    public function getDashboardStats(): array
    {
        $stats = [
            'total_leads'      => 0,
            'total_forms'      => 0,
            'leads_today'      => 0,
            'high_score_leads' => 0,
            'leads_this_week'  => 0,
        ];

        try {
            // Total leads
            $stmt               = $this->pdo->query("SELECT COUNT(*) AS count FROM leads");
            $stats['total_leads'] = (int) $stmt->fetch(PDO::FETCH_ASSOC)['count'];

            // Total forms
            $stmt               = $this->pdo->query("SELECT COUNT(*) AS count FROM forms");
            $stats['total_forms'] = (int) $stmt->fetch(PDO::FETCH_ASSOC)['count'];

            // Leads today
            $stmt                 = $this->pdo->query("SELECT COUNT(*) AS count FROM leads WHERE DATE(created_at) = CURDATE()");
            $stats['leads_today'] = (int) $stmt->fetch(PDO::FETCH_ASSOC)['count'];

            // High score leads (>= 80)
            $stmt                     = $this->pdo->query("SELECT COUNT(*) AS count FROM leads WHERE score >= 80");
            $stats['high_score_leads'] = (int) $stmt->fetch(PDO::FETCH_ASSOC)['count'];

            // Leads this week
            $stmt                    = $this->pdo->query("SELECT COUNT(*) AS count FROM leads WHERE YEARWEEK(created_at) = YEARWEEK(NOW())");
            $stats['leads_this_week'] = (int) $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        } catch (PDOException $e) {
            // Log error but don't break the application
            error_log("Dashboard stats error: " . $e->getMessage());
        }

        return $stats;
    }

    /**
     * Get recent leads (default last 10)
     */
    public function getRecentLeads(int $limit = 10): array
    {
        $sql = "
            SELECT l.*, f.name AS form_name 
            FROM leads l 
            LEFT JOIN forms f ON l.form_id = f.id 
            ORDER BY l.created_at DESC 
            LIMIT ?
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$limit]);

        $leads = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $leads[] = new Lead($row);
        }

        return $leads;
    }

    /**
     * Delete a form and its associated leads
     */
    public function deleteForm(int $formId): bool
    {
        $this->pdo->beginTransaction();

        try {
            // Delete associated leads first
            $stmt = $this->pdo->prepare("DELETE FROM leads WHERE form_id = ?");
            $stmt->execute([$formId]);

            // Delete the form
            $stmt = $this->pdo->prepare("DELETE FROM forms WHERE id = ?");
            $stmt->execute([$formId]);

            $this->pdo->commit();
            return true;
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Form deletion error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Search leads by email, name, or company via JSON data field
     */
    public function searchLeads(string $query): array
    {
        $sql = "
            SELECT l.*, f.name AS form_name 
            FROM leads l 
            LEFT JOIN forms f ON l.form_id = f.id 
            WHERE l.data LIKE ? 
            ORDER BY l.created_at DESC
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['%' . $query . '%']);

        $leads = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $leads[] = new Lead($row);
        }

        return $leads;
    }
}
