<?php
// SkyBlue Form Builder (Create/Edit)
// Assumes this view is rendered inside your SkyBlue main layout
// Controller should pass: $form, $defaultStructure, $fieldTypes
?>

<div
    class="dashboard-container"
    x-data="formBuilder()"
    x-init="init(<?= htmlspecialchars(json_encode($defaultStructure)) ?>)"
>
    <!-- Page Header -->
    <header class="dashboard-header">
        <div class="header-content">
            <div class="welcome-section">
                <h1>
                    <?= $form
                        ? 'Edit Form: ' . htmlspecialchars($form['name'])
                        : 'Create New Form' ?>
                </h1>
                <p>Drag & drop fields to build your lead capture form.</p>
            </div>
            <div class="header-actions" style="display:flex; gap:0.75rem;">
                <button
                    id="previewButton"
                    type="button"
                    @click="openPreview()"
                    class="btn-primary"
                    style="background:var(--sky-500); display:inline-flex; align-items:center; gap:.5rem;"
                >
                    <i class="fas fa-eye"></i>
                    Preview
                </button>
                <button
                    type="button"
                    @click="saveForm()"
                    class="btn-primary"
                    style="display:inline-flex; align-items:center; gap:.5rem;"
                >
                    <i class="fas fa-save"></i>
                    Save Form
                </button>
            </div>
        </div>

        <!-- Optional quick hints -->
        <div class="quick-stats" style="margin-top:1.5rem;">
            <div class="stat-card">
                <div class="stat-number">
                    <i class="fas fa-mouse-pointer"></i>
                </div>
                <div class="stat-label">Drag fields from the left</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">
                    <i class="fas fa-sliders-h"></i>
                </div>
                <div class="stat-label">Configure fields on the right</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">
                    <i class="fas fa-eye"></i>
                </div>
                <div class="stat-label">Preview before saving</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">
                    <i class="fas fa-robot"></i>
                </div>
                <div class="stat-label">Auto-create & score leads</div>
            </div>
        </div>
    </header>

    <!-- Main Builder Grid -->
    <div class="dashboard-grid" style="grid-template-columns: 1.3fr 2fr 1.3fr; align-items:flex-start;">
        <!-- LEFT: Field Palette + Form Settings -->
        <div class="left-column">
            <!-- Field Palette -->
            <section class="tasks-section">
                <div class="section-header">
                    <h2>
                        <i class="fas fa-layer-group"></i>
                        Form Fields
                    </h2>
                    <p style="font-size:.85rem; color:var(--sky-700); margin-top:.25rem;">
                        Drag fields from here into the canvas.
                    </p>
                </div>
                <div class="field-palette-list" style="display:flex; flex-direction:column; gap:.75rem;">
                    <?php foreach ($fieldTypes as $type => $config): ?>
                        <div
                            class="field-palette-item"
                            draggable="true"
                            @dragstart="dragStart($event, '<?= $type ?>')"
                        >
                            <div class="field-palette-inner">
                                <div class="field-palette-icon">
                                    <?= strtoupper(substr($type, 0, 1)) ?>
                                </div>
                                <div>
                                    <div class="field-palette-name">
                                        <?= htmlspecialchars($config['name']) ?>
                                    </div>
                                    <?php if (!empty($config['description'])): ?>
                                        <div class="field-palette-desc">
                                            <?= htmlspecialchars($config['description']) ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>

            <!-- Form Settings -->
            <section class="reports-section">
                <div class="section-header">
                    <h2>
                        <i class="fas fa-sliders-h"></i>
                        Form Settings
                    </h2>
                </div>
                <div class="settings-grid" style="display:flex; flex-direction:column; gap:1rem;">
                    <div>
                        <label class="sky-label">Form Name</label>
                        <input
                            type="text"
                            id="form-name"
                            x-model="formName"
                            class="sky-input"
                            placeholder="Lead capture form"
                        >
                    </div>

                    <div>
                        <label class="sky-label">Description</label>
                        <textarea
                            id="form-description"
                            x-model="formDescription"
                            rows="3"
                            class="sky-textarea"
                            placeholder="Short description for internal use"
                        ></textarea>
                    </div>

                    <div>
                        <label class="sky-label">Success Message</label>
                        <input
                            type="text"
                            id="success-message"
                            x-model="settings.success_message"
                            class="sky-input"
                            placeholder="Thank you for your submission!"
                        >
                    </div>

                    <div>
                        <label class="sky-label">Submit Button Text</label>
                        <input
                            type="text"
                            id="submit-text"
                            x-model="settings.submit_text"
                            class="sky-input"
                            placeholder="Submit"
                        >
                    </div>

                    <label class="sky-checkbox">
                        <input
                            id="auto-create-lead"
                            type="checkbox"
                            x-model="settings.auto_create_lead"
                        >
                        <span>Automatically create leads from submissions</span>
                    </label>

                    <label class="sky-checkbox">
                        <input
                            id="auto-ai-scoring"
                            type="checkbox"
                            x-model="settings.auto_ai_scoring"
                        >
                        <span>Auto-score new leads with AI</span>
                    </label>
                </div>
            </section>
        </div>

        <!-- CENTER: Form Canvas -->
        <div class="center-column">
            <section class="tasks-section">
                <div class="section-header">
                    <h2>
                        <i class="fas fa-object-group"></i>
                        Form Canvas
                    </h2>
                    <p style="font-size:.85rem; color:var(--sky-700); margin-top:.25rem;">
                        Arrange and configure your fields here.
                    </p>
                </div>

                <div
                    class="form-canvas sky-canvas"
                    @drop="drop($event)"
                    @dragover="allowDrop($event)"
                >
                    <!-- Empty State -->
                    <div x-show="fields.length === 0" class="canvas-empty">
                        <div class="canvas-empty-icon">
                            <i class="fas fa-mouse-pointer"></i>
                        </div>
                        <h3>Start building your form</h3>
                        <p>Drag fields from the left panel into this canvas.</p>
                    </div>

                    <!-- Fields List -->
                    <div x-show="fields.length > 0" class="canvas-fields" style="display:flex; flex-direction:column; gap:1rem;">
                        <template x-for="(field, index) in fields" :key="field.id">
                            <div
                                class="field-item sky-field-item"
                                :data-field-id="field.id"
                                @click="selectField(index)"
                                :class="{'sky-field-item-active': selectedFieldIndex === index}"
                            >
                                <div class="field-item-header">
                                    <div>
                                        <div class="field-item-title">
                                            <span x-text="field.label || 'Untitled Field'"></span>
                                            <span x-show="field.required" class="required-badge">*</span>
                                        </div>
                                        <div class="field-item-meta">
                                            <span x-text="field.type"></span>
                                        </div>
                                    </div>
                                    <div class="field-item-actions">
                                        <button
                                            type="button"
                                            @click.stop="moveField(index, -1)"
                                            :disabled="index === 0"
                                            title="Move up"
                                        >
                                            <i class="fas fa-arrow-up"></i>
                                        </button>
                                        <button
                                            type="button"
                                            @click.stop="moveField(index, 1)"
                                            :disabled="index === fields.length - 1"
                                            title="Move down"
                                        >
                                            <i class="fas fa-arrow-down"></i>
                                        </button>
                                        <button
                                            type="button"
                                            @click.stop="removeField(index)"
                                            class="danger"
                                            title="Remove field"
                                        >
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>

                                <!-- Preview -->
                                <div class="field-preview">
                                    <template x-if="['text','email','phone','number'].includes(field.type)">
                                        <input
                                            :type="field.type"
                                            :placeholder="field.placeholder || ''"
                                            class="sky-input"
                                            disabled
                                        >
                                    </template>

                                    <template x-if="field.type === 'textarea'">
                                        <textarea
                                            :placeholder="field.placeholder || ''"
                                            class="sky-textarea"
                                            rows="3"
                                            disabled
                                        ></textarea>
                                    </template>

                                    <template x-if="field.type === 'select'">
                                        <select class="sky-input" disabled>
                                            <option>Select an option</option>
                                            <template x-for="option in field.options || []">
                                                <option :value="option.value" x-text="option.label"></option>
                                            </template>
                                        </select>
                                    </template>

                                    <template x-if="['checkbox','radio'].includes(field.type)">
                                        <div class="field-options">
                                            <template x-for="option in field.options || []">
                                                <label class="sky-option">
                                                    <input
                                                        :type="field.type"
                                                        disabled
                                                    >
                                                    <span x-text="option.label"></span>
                                                </label>
                                            </template>
                                        </div>
                                    </template>

                                    <template x-if="field.type === 'date'">
                                        <input type="date" class="sky-input" disabled>
                                    </template>

                                    <template x-if="field.type === 'file'">
                                        <input type="file" class="sky-input" disabled>
                                    </template>

                                    <p
                                        x-show="field.description"
                                        class="field-description"
                                        x-text="field.description"
                                    ></p>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>
            </section>
        </div>

        <!-- RIGHT: Field Properties -->
        <div class="right-column">
            <section
                class="quick-actions-section"
                x-show="selectedField !== null"
                x-transition
            >
                <div class="section-header">
                    <h2>
                        <i class="fas fa-wrench"></i>
                        Field Properties
                    </h2>
                    <p
                        style="font-size:.85rem; color:var(--sky-700); margin-top:.25rem;"
                        x-text="getFieldTypeName(selectedField?.type || '')"
                    ></p>
                </div>

                <div class="settings-grid" style="display:flex; flex-direction:column; gap:1rem;">
                    <div>
                        <label class="sky-label">Field Label</label>
                        <input
                            type="text"
                            x-model="selectedField.label"
                            @input="updateFieldProperty('label', $event.target.value)"
                            class="sky-input"
                        >
                    </div>

                    <div>
                        <label class="sky-label">Field ID</label>
                        <input
                            type="text"
                            x-model="selectedField.id"
                            class="sky-input"
                            placeholder="field_name"
                        >
                        <p class="sky-help">Unique identifier used in submissions.</p>
                    </div>

                    <div>
                        <label class="sky-label">Placeholder Text</label>
                        <input
                            type="text"
                            x-model="selectedField.placeholder"
                            @input="updateFieldProperty('placeholder', $event.target.value)"
                            class="sky-input"
                        >
                    </div>

                    <div>
                        <label class="sky-label">Description</label>
                        <textarea
                            x-model="selectedField.description"
                            @input="updateFieldProperty('description', $event.target.value)"
                            class="sky-textarea"
                            rows="2"
                        ></textarea>
                    </div>

                    <label class="sky-checkbox">
                        <input
                            type="checkbox"
                            x-model="selectedField.required"
                            @change="updateFieldProperty('required', $event.target.checked)"
                        >
                        <span>Required field</span>
                    </label>

                    <!-- Options for select, radio, checkbox -->
                    <template x-if="['select','radio','checkbox'].includes(selectedField.type)">
                        <div>
                            <label class="sky-label">Options</label>
                            <div class="options-list">
                                <template x-for="(option, optionIndex) in selectedField.options" :key="optionIndex">
                                    <div class="option-row">
                                        <input
                                            type="text"
                                            x-model="option.label"
                                            placeholder="Option label"
                                            class="sky-input"
                                        >
                                        <input
                                            type="text"
                                            x-model="option.value"
                                            placeholder="Value"
                                            class="sky-input"
                                        >
                                        <button
                                            type="button"
                                            @click="removeOption(optionIndex)"
                                            class="option-remove"
                                            title="Remove option"
                                        >
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </template>
                                <button
                                    type="button"
                                    @click="addOption()"
                                    class="btn-primary"
                                    style="width:100%; justify-content:center; display:inline-flex; align-items:center; gap:.5rem; background:var(--sky-500);"
                                >
                                    <i class="fas fa-plus"></i>
                                    Add Option
                                </button>
                            </div>
                        </div>
                    </template>

                    <!-- File settings -->
                    <template x-if="selectedField.type === 'file'">
                        <div>
                            <label class="sky-label">Accepted File Types</label>
                            <input
                                type="text"
                                x-model="selectedField.accept"
                                @input="updateFieldProperty('accept', $event.target.value)"
                                class="sky-input"
                                placeholder=".pdf,.doc,.docx,.jpg,.png"
                            >
                            <p class="sky-help">Comma-separated file extensions.</p>
                        </div>
                    </template>
                </div>
            </section>
        </div>
    </div>

    <!-- Preview Modal -->
    <div
        class="form-preview-overlay"
        x-show="showPreview"
        x-transition.opacity
        @keydown.window.escape="closePreview()"
    >
        <div class="form-preview-backdrop" @click="closePreview()"></div>

        <div
            class="tasks-section form-preview-modal"
            @click.stop
        >
            <div class="section-header" style="display:flex; justify-content:space-between; align-items:center;">
                <h2>
                    <i class="fas fa-eye"></i>
                    Form Preview
                </h2>
                <button
                    type="button"
                    @click="closePreview()"
                    class="btn-primary"
                    style="background:var(--danger); display:inline-flex; align-items:center; gap:.4rem; padding:.4rem .9rem;"
                >
                    <i class="fas fa-times"></i>
                    Close
                </button>
            </div>

            <div class="preview-content" style="background:var(--sky-50); padding:1.25rem; border-radius:12px;">
                <form class="space-y-4">
                    <template x-for="field in fields" :key="field.id">
                        <div class="preview-field">
                            <label class="sky-label">
                                <span x-text="field.label"></span>
                                <span x-show="field.required" class="required-badge">*</span>
                            </label>

                            <template x-if="['text','email','phone','number'].includes(field.type)">
                                <input
                                    :type="field.type"
                                    :placeholder="field.placeholder || ''"
                                    class="sky-input"
                                >
                            </template>

                            <template x-if="field.type === 'textarea'">
                                <textarea
                                    :placeholder="field.placeholder || ''"
                                    class="sky-textarea"
                                    rows="3"
                                ></textarea>
                            </template>

                            <template x-if="field.type === 'select'">
                                <select class="sky-input">
                                    <option value="">Select an option</option>
                                    <template x-for="option in field.options || []">
                                        <option :value="option.value" x-text="option.label"></option>
                                    </template>
                                </select>
                            </template>

                            <template x-if="field.type === 'checkbox'">
                                <div class="field-options">
                                    <template x-for="option in field.options || []">
                                        <label class="sky-option">
                                            <input type="checkbox" :value="option.value">
                                            <span x-text="option.label"></span>
                                        </label>
                                    </template>
                                </div>
                            </template>

                            <template x-if="field.type === 'radio'">
                                <div class="field-options">
                                    <template x-for="option in field.options || []">
                                        <label class="sky-option">
                                            <input type="radio" :value="option.value" :name="field.id">
                                            <span x-text="option.label"></span>
                                        </label>
                                    </template>
                                </div>
                            </template>

                            <template x-if="field.type === 'date'">
                                <input type="date" class="sky-input">
                            </template>

                            <template x-if="field.type === 'file'">
                                <input type="file" :accept="field.accept || ''" class="sky-input">
                            </template>

                            <p
                                x-show="field.description"
                                class="field-description"
                                x-text="field.description"
                            ></p>
                        </div>
                    </template>

                    <button
                        type="submit"
                        class="btn-primary"
                        style="width:100%; justify-content:center; display:inline-flex; align-items:center; gap:.5rem;"
                    >
                        <span x-text="settings.submit_text || 'Submit'"></span>
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
function formBuilder() {
    return {
        formName: '<?= $form ? htmlspecialchars($form['name']) : "New Form" ?>',
        formDescription: '<?= $form ? htmlspecialchars($form['description']) : "" ?>',
        fields: [],
        selectedField: null,
        selectedFieldIndex: null,
        showPreview: false,
        settings: {
            success_message: 'Thank you for your submission!',
            submit_text: 'Submit',
            auto_create_lead: true,
            auto_ai_scoring: true
        },
        fieldTypes: <?= json_encode($fieldTypes) ?>,

        init(defaultStructure) {
            // Always start with preview closed
            this.showPreview = false;

            if (defaultStructure && defaultStructure.fields) {
                this.fields = defaultStructure.fields;
                this.settings = { ...this.settings, ...defaultStructure.settings };
            }

            <?php if ($form): ?>
            this.formName = '<?= htmlspecialchars($form['name']) ?>';
            this.formDescription = '<?= htmlspecialchars($form['description']) ?>';
            <?php endif; ?>
        },

        dragStart(event, fieldType) {
            event.dataTransfer.setData('application/json', JSON.stringify({
                type: fieldType,
                id: 'field_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9)
            }));
        },

        allowDrop(event) {
            event.preventDefault();
        },

        drop(event) {
            event.preventDefault();
            const data = JSON.parse(event.dataTransfer.getData('application/json'));

            const newField = {
                id: data.id,
                type: data.type,
                label: this.getFieldTypeName(data.type),
                required: false,
                placeholder: '',
                description: ''
            };

            if (['select', 'radio', 'checkbox'].includes(data.type)) {
                newField.options = [
                    { label: 'Option 1', value: 'option1' },
                    { label: 'Option 2', value: 'option2' }
                ];
            }

            this.fields.push(newField);
            this.selectField(this.fields.length - 1);
        },

        selectField(index) {
            this.selectedFieldIndex = index;
            this.selectedField = this.fields[index];
        },

        updateFieldProperty(property, value) {
            if (this.selectedFieldIndex !== null) {
                this.fields[this.selectedFieldIndex][property] = value;
            }
        },

        removeField(index) {
            this.fields.splice(index, 1);
            if (this.selectedFieldIndex === index) {
                this.selectedField = null;
                this.selectedFieldIndex = null;
            } else if (this.selectedFieldIndex > index) {
                this.selectedFieldIndex--;
            }
        },

        moveField(index, direction) {
            const newIndex = index + direction;
            if (newIndex >= 0 && newIndex < this.fields.length) {
                const field = this.fields.splice(index, 1)[0];
                this.fields.splice(newIndex, 0, field);
                this.selectedFieldIndex = newIndex;
                this.selectedField = this.fields[newIndex];
            }
        },

        addOption() {
            if (this.selectedField && this.selectedField.options) {
                this.selectedField.options.push({
                    label: 'New Option',
                    value: 'new_option_' + Date.now()
                });
            }
        },

        removeOption(index) {
            if (this.selectedField && this.selectedField.options) {
                this.selectedField.options.splice(index, 1);
            }
        },

        getFieldTypeName(type) {
            return this.fieldTypes[type]?.name || type;
        },

        openPreview() {
            // Only open preview when explicitly requested
            this.showPreview = true;
        },

        closePreview() {
            this.showPreview = false;
        },

        async saveForm() {
            const formStructure = {
                fields: this.fields,
                settings: this.settings,
                version: '1.0'
            };

            const formData = new FormData();
            formData.append('name', this.formName);
            formData.append('description', this.formDescription);
            formData.append('structure', JSON.stringify(formStructure));
            formData.append('settings', JSON.stringify(this.settings));

            try {
                const response = await fetch('<?= $form ? "/lead-management/public/app/forms/{$form['id']}/edit" : "/lead-management/public/app/forms/create" ?>', {
                    method: 'POST',
                    body: formData
                });

                if (response.ok) {
                    window.location.href = '/lead-management/public/app/forms';
                } else {
                    alert('Error saving form');
                }
            } catch (error) {
                alert('Error saving form: ' + error.message);
            }
        }
    }
}
</script>

<style>
/* Overlay for preview */
.form-preview-overlay {
    position: fixed;
    inset: 0;
    z-index: 1000;
    display: none;
}
.form-preview-overlay[x-cloak],
.form-preview-overlay[style*="display: none"] {
    display: none !important;
}
.form-preview-overlay[x-show="showPreview"] {
    display: block;
}
.form-preview-backdrop {
    position: absolute;
    inset: 0;
    background: rgba(15, 23, 42, 0.55);
}
.form-preview-modal {
    position: relative;
    max-width: 650px;
    width: 100%;
    max-height: 80vh;
    margin: 4rem auto;
    overflow: auto;
    z-index: 1001;
    border-radius: 18px;
    box-shadow: 0 20px 45px rgba(15, 23, 42, 0.45);
}

/* SkyBlue-specific tweaks for builder */

/* Palette items */
.field-palette-item {
    background: linear-gradient(135deg, #ffffff, #f0f9ff);
    border-radius: 10px;
    padding: .75rem;
    border: 1px solid var(--sky-200);
    cursor: grab;
    box-shadow: var(--shadow);
    transition: var(--transition);
}
.field-palette-item:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow-lg);
    border-color: var(--primary);
}
.field-palette-inner {
    display:flex;
    align-items:center;
    gap:.75rem;
}
.field-palette-icon {
    width:36px;
    height:36px;
    border-radius:10px;
    background:var(--sky-50);
    border:1px solid var(--sky-200);
    display:flex;
    align-items:center;
    justify-content:center;
    font-weight:700;
    color:var(--sky-700);
}
.field-palette-name {
    font-weight:600;
    color:var(--sky-800);
}
.field-palette-desc {
    font-size:.8rem;
    color:var(--sky-700);
}

/* Canvas */
.sky-canvas {
    min-height: 420px;
    background: var(--sky-50);
    border-radius: 16px;
    border: 2px dashed var(--sky-300);
    padding: 1.5rem;
    transition: var(--transition);
}
.canvas-empty {
    text-align:center;
    padding:3rem 1rem;
    color:var(--sky-700);
}
.canvas-empty-icon {
    width:56px;
    height:56px;
    border-radius:18px;
    background:rgba(59,130,246,0.08);
    color:var(--primary);
    display:grid;
    place-items:center;
    font-size:1.4rem;
    margin:0 auto .75rem;
}

/* Field cards */
.sky-field-item {
    background:#ffffff;
    border-radius:14px;
    border:1px solid var(--sky-200);
    box-shadow:var(--shadow);
    padding:1rem;
    transition:var(--transition);
}
.sky-field-item:hover {
    transform:translateY(-2px);
    box-shadow:var(--shadow-lg);
    border-color:var(--primary-light);
}
.sky-field-item-active {
    border-color:var(--primary);
    box-shadow:0 0 0 2px rgba(59,130,246,0.15);
}
.field-item-header {
    display:flex;
    justify-content:space-between;
    align-items:flex-start;
    gap:1rem;
    margin-bottom:.75rem;
}
.field-item-title {
    font-weight:600;
    color:var(--sky-800);
    display:flex;
    align-items:center;
    gap:.25rem;
}
.field-item-meta {
    font-size:.8rem;
    color:var(--sky-600);
}
.required-badge {
    color:var(--danger);
    font-weight:700;
}
.field-item-actions {
    display:flex;
    gap:.25rem;
}
.field-item-actions button {
    border:none;
    background:transparent;
    padding:.25rem .45rem;
    border-radius:999px;
    cursor:pointer;
    color:var(--sky-600);
    transition:var(--transition);
    font-size:.8rem;
}
.field-item-actions button:hover {
    background:var(--sky-100);
    color:var(--primary);
}
.field-item-actions button.danger:hover {
    background:rgba(239,68,68,.08);
    color:var(--danger);
}

/* Inputs */
.sky-label {
    display:block;
    font-size:.85rem;
    font-weight:600;
    color:var(--sky-800);
    margin-bottom:.25rem;
}
.sky-input,
.sky-textarea,
.preview-content input,
.preview-content textarea,
.preview-content select {
    width:100%;
    border-radius:10px;
    border:1px solid var(--sky-200);
    padding:.55rem .75rem;
    font-size:.88rem;
    outline:none;
    transition:var(--transition);
    background:#fff;
}
.sky-input:focus,
.sky-textarea:focus,
.preview-content input:focus,
.preview-content textarea:focus,
.preview-content select:focus {
    border-color:var(--primary);
    box-shadow:0 0 0 2px rgba(59,130,246,0.15);
}
.sky-textarea {
    resize:vertical;
}
.sky-checkbox {
    display:flex;
    align-items:flex-start;
    gap:.5rem;
    font-size:.85rem;
    color:var(--sky-800);
}
.sky-checkbox input {
    margin-top:.15rem;
}
.sky-help {
    font-size:.78rem;
    color:var(--sky-700);
    margin-top:.15rem;
}

/* Options editor */
.options-list {
    display:flex;
    flex-direction:column;
    gap:.5rem;
}
.option-row {
    display:grid;
    grid-template-columns:1fr 1fr auto;
    gap:.4rem;
    align-items:center;
}
.option-remove {
    border:none;
    background:transparent;
    color:var(--danger);
    padding:.25rem .45rem;
    border-radius:999px;
    cursor:pointer;
    transition:var(--transition);
}
.option-remove:hover {
    background:rgba(239,68,68,.08);
}

/* Choice preview */
.field-options {
    display:flex;
    flex-direction:column;
    gap:.25rem;
    margin-top:.25rem;
}
.sky-option {
    display:flex;
    align-items:center;
    gap:.4rem;
    font-size:.85rem;
    color:var(--sky-800);
}
.field-description {
    font-size:.8rem;
    color:var(--sky-700);
    margin-top:.35rem;
}
</style>
