// ====== Helper Functions ======

import { debounce } from './utils.js';

/**
 * DOM ready helper
 */
export function domReady(callback) {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', callback);
    } else {
        callback();
    }
}

/**
 * Query selector helper with optional context
 */
export function $(selector, context = document) {
    return context.querySelector(selector);
}

/**
 * Query selector all helper with optional context
 */
export function $$(selector, context = document) {
    return context.querySelectorAll(selector);
}

/**
 * Create element with attributes and children
 */
export function createElement(tag, attributes = {}, children = []) {
    const element = document.createElement(tag);
    
    // Set attributes
    Object.keys(attributes).forEach(key => {
        if (key === 'className') {
            element.className = attributes[key];
        } else if (key === 'htmlFor') {
            element.htmlFor = attributes[key];
        } else if (key.startsWith('on') && typeof attributes[key] === 'function') {
            element.addEventListener(key.slice(2).toLowerCase(), attributes[key]);
        } else {
            element.setAttribute(key, attributes[key]);
        }
    });
    
    // Append children
    if (typeof children === 'string') {
        element.textContent = children;
    } else if (Array.isArray(children)) {
        children.forEach(child => {
            if (typeof child === 'string') {
                element.appendChild(document.createTextNode(child));
            } else if (child instanceof Node) {
                element.appendChild(child);
            }
        });
    }
    
    return element;
}

/**
 * Show notification/toast
 */
export function showNotification(message, type = 'info', duration = 5000) {
    // Remove existing notifications
    const existing = document.querySelector('.notification');
    if (existing) existing.remove();
    
    const notification = createElement('div', {
        className: `notification notification--${type}`
    }, message);
    
    document.body.appendChild(notification);
    
    // Add show class after a frame
    requestAnimationFrame(() => {
        notification.classList.add('notification--show');
    });
    
    // Auto remove
    if (duration > 0) {
        setTimeout(() => {
            notification.classList.remove('notification--show');
            setTimeout(() => notification.remove(), 300);
        }, duration);
    }
    
    return notification;
}

/**
 * Toggle loading state
 */
export function setLoadingState(element, isLoading) {
    if (isLoading) {
        element.setAttribute('data-loading', 'true');
        element.disabled = true;
    } else {
        element.removeAttribute('data-loading');
        element.disabled = false;
    }
}

/**
 * Format file size
 */
export function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

/**
 * Sanitize HTML
 */
export function sanitizeHtml(html) {
    const temp = document.createElement('div');
    temp.textContent = html;
    return temp.innerHTML;
}